// Chat Widget Script
(function() {
    // ---------- helpers ----------
    function deepMerge(target, source) {
        if (!source || typeof source !== 'object') return target;
        const out = Array.isArray(target) ? target.slice() : { ...target };
        for (const key of Object.keys(source)) {
            const sv = source[key];
            const tv = out[key];
            if (sv && typeof sv === 'object' && !Array.isArray(sv)) {
                out[key] = deepMerge(tv && typeof tv === 'object' ? tv : {}, sv);
            } else {
                out[key] = Array.isArray(sv) ? sv.slice() : sv;
            }
        }
        return out;
    }

    function safeGet(obj, path, fallback) {
        try {
            return path.split('.').reduce((o, k) => (o && k in o ? o[k] : undefined), obj) ?? fallback;
        } catch (_) {
            return fallback;
        }
    }

    // ---------- styles ----------
    const styles = `
        .n8n-chat-widget {
            --chat--color-primary: var(--n8n-chat-primary-color, #002657);
            --chat--color-secondary: var(--n8n-chat-secondary-color, #FA4616);
            --chat--color-background: var(--n8n-chat-background-color, #ffffff);
            --chat--color-font: var(--n8n-chat-font-color, #333333);
            font-family: 'Geist Sans', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
        }
        .n8n-chat-widget .chat-container {
            position: fixed; bottom: 20px; right: 20px; z-index: 1000;
            display: none; width: 380px; height: 600px;
            background: var(--chat--color-background); border-radius: 12px;
            box-shadow: 0 8px 32px rgba(133, 79, 255, 0.15);
            border: 1px solid rgba(133, 79, 255, 0.2); overflow: hidden;
            font-family: inherit;
        }
        .n8n-chat-widget .chat-container.position-left { right: auto; left: 20px; }
        .n8n-chat-widget .chat-container.open { display: flex; flex-direction: column; }
        .n8n-chat-widget .brand-header { padding: 16px; display: flex; align-items: center; gap: 12px; border-bottom: 1px solid rgba(133, 79, 255, 0.1); position: relative; }
        .n8n-chat-widget .close-button {
            position: absolute; right: 16px; top: 50%; transform: translateY(-50%);
            background: none; border: none; cursor: pointer;
            font-size: 22px; color: var(--chat--color-font); opacity: 0.6;
            width: 40px; height: 40px; display: flex; align-items: center; justify-content: center;
            border-radius: 50%; box-shadow: none;
        }
        .n8n-chat-widget .close-button:hover { opacity: 1; color: #FA4616; }
        .n8n-chat-widget .brand-header img { width: 30px; height: 30px; }
        .n8n-chat-widget .brand-header span { font-size: 18px; font-weight: 500; color: var(--chat--color-font); }
        .n8n-chat-widget .new-conversation { position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); padding: 20px; text-align: center; width: 100%; max-width: 300px; }
        .n8n-chat-widget .welcome-text { font-size: 24px; font-weight: 600; color: var(--chat--color-font); margin-bottom: 24px; }
        .n8n-chat-widget .new-chat-btn { width: 100%; padding: 16px 24px; background: linear-gradient(135deg, var(--chat--color-primary) 0%, var(--chat--color-secondary) 100%); color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 16px; font-weight: 500; margin-bottom: 12px; transition: transform .2s; }
        .n8n-chat-widget .new-chat-btn:hover { transform: scale(1.02); }
        .n8n-chat-widget .chat-interface { display: none; flex-direction: column; height: 100%; }
        .n8n-chat-widget .chat-interface.active { display: flex; }
        .n8n-chat-widget .chat-messages { flex: 1; overflow-y: auto; padding: 20px; display: flex; flex-direction: column; }
        .n8n-chat-widget .chat-message { padding: 12px 16px; margin: 8px 0; border-radius: 12px; max-width: 80%; font-size: 14px; line-height: 1.5; }
        .n8n-chat-widget .chat-message.user { background: linear-gradient(135deg, var(--chat--color-primary) 0%, var(--chat--color-secondary) 100%); color: white; align-self: flex-end; }
        .n8n-chat-widget .chat-message.bot { background: var(--chat--color-background); border: 1px solid rgba(133, 79, 255, 0.2); align-self: flex-start; }
        .n8n-chat-widget .chat-message.bot strong { font-weight: bold; color: #002657; }
        .n8n-chat-widget .chat-message.bot em { font-style: italic; color: #333; }
        .n8n-chat-widget .chat-message.bot a { color: #002657; text-decoration: underline; }
        /* Clean, tight lists */
        .n8n-chat-widget .chat-message.bot p { margin: 6px 0; }
        .n8n-chat-widget .chat-message.bot ul,
        .n8n-chat-widget .chat-message.bot ol {
            padding-left: 24px;
            margin: 6px 0;
            list-style-position: outside;
        }
        .n8n-chat-widget .chat-message.bot li {
            margin: 2px 0;
            line-height: 1.4;
        }
        .n8n-chat-widget .chat-input { padding: 16px; border-top: 1px solid rgba(133, 79, 255, 0.1); display: flex; gap: 8px; }
        .n8n-chat-widget .chat-input textarea { flex: 1; padding: 12px; border: 1px solid rgba(133, 79, 255, 0.2); border-radius: 8px; resize: none; }

        /* Send button = same style as New Chat */
        .n8n-chat-widget .chat-input button {
            padding: 12px 20px;
            background: linear-gradient(135deg, var(--chat--color-primary) 0%, var(--chat--color-secondary) 100%);
            color: white;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 500;
            transition: transform 0.2s;
        }
        .n8n-chat-widget .chat-input button:hover { transform: scale(1.05); }

        /* Footer branding */
        .n8n-chat-widget .chat-footer {
            padding: 8px;
            text-align: center;
            background: var(--chat--color-background);
            border-top: 1px solid rgba(133, 79, 255, 0.1);
        }
        .n8n-chat-widget .chat-footer a {
            font-size: 12px;
            opacity: 0.9;
            display: inline-block;
            color: #002657;
            font-weight: 500;
            text-decoration: none;
        }
        .n8n-chat-widget .chat-footer a:hover { opacity: 1; text-decoration: underline; }

        /* Toggle button */
        .n8n-chat-widget .chat-toggle {
            position: fixed; bottom: 20px; right: 20px; width: 60px; height: 60px;
            border-radius: 30px; background: linear-gradient(135deg, var(--chat--color-primary) 0%, var(--chat--color-secondary) 100%);
            color: white; border: none; cursor: pointer; z-index: 999;
            display: flex; align-items: center; justify-content: center;
        }
        .n8n-chat-widget .chat-toggle.position-left { right: auto; left: 20px; }
        .n8n-chat-widget .chat-toggle svg { width: 24px; height: 24px; }
    `;

    // Load font once
    if (!document.querySelector('link[href*="geist-sans"]')) {
        const fontLink = document.createElement('link');
        fontLink.rel = 'stylesheet';
        fontLink.href = 'https://cdn.jsdelivr.net/npm/geist@1.0.0/dist/fonts/geist-sans/style.css';
        document.head.appendChild(fontLink);
    }

    // Inject styles once
    if (!document.getElementById('n8n-chat-widget-css')) {
        const styleSheet = document.createElement('style');
        styleSheet.id = 'n8n-chat-widget-css';
        styleSheet.textContent = styles;
        document.head.appendChild(styleSheet);
    }

    // ---------- configuration ----------
    const defaultConfig = {
        webhook: { url: '', route: '' },
        branding: {
            logo: '',
            name: 'Chat Assistant',
            welcomeText: 'Welcome!',
            responseTimeText: 'Typically replies instantly',
            poweredBy: { text: 'Powered by legalGPTs', link: 'https://legalgpts.com' }
        },
        style: {
            primaryColor: '#854fff',
            secondaryColor: '#6b3fd4',
            position: 'right',
            backgroundColor: '#ffffff',
            fontColor: '#333333'
        }
    };

    const userConfig = (typeof window !== 'undefined' && window.ChatWidgetConfig) ? window.ChatWidgetConfig : {};
    const config = deepMerge(defaultConfig, userConfig);

    // ---------- build widget ----------
    if (window.N8NChatWidgetInitialized) return;
    window.N8NChatWidgetInitialized = true;

    const widgetContainer = document.createElement('div');
    widgetContainer.className = 'n8n-chat-widget';

    // Apply theme variables from config
    widgetContainer.style.setProperty('--n8n-chat-primary-color', safeGet(config, 'style.primaryColor', defaultConfig.style.primaryColor));
    widgetContainer.style.setProperty('--n8n-chat-secondary-color', safeGet(config, 'style.secondaryColor', defaultConfig.style.secondaryColor));
    widgetContainer.style.setProperty('--n8n-chat-background-color', safeGet(config, 'style.backgroundColor', defaultConfig.style.backgroundColor));
    widgetContainer.style.setProperty('--n8n-chat-font-color', safeGet(config, 'style.fontColor', defaultConfig.style.fontColor));

    const chatContainer = document.createElement('div');
    chatContainer.className = `chat-container${safeGet(config, 'style.position', 'right') === 'left' ? ' position-left' : ''}`;

    const newConversationHTML = `
        <div class="brand-header">
            <img src="${safeGet(config, 'branding.logo', '')}" alt="${safeGet(config, 'branding.name', 'Chat Assistant')}">
            <span>${safeGet(config, 'branding.name', 'Chat Assistant')}</span>
            <button class="close-button">×</button>
        </div>
        <div class="new-conversation">
            <h2 class="welcome-text">${safeGet(config, 'branding.welcomeText', 'Welcome!')}</h2>
            <button class="new-chat-btn">Send us a message</button>
            <p class="response-text">${safeGet(config, 'branding.responseTimeText', 'Typically replies instantly')}</p>
        </div>
    `;

    const chatInterfaceHTML = `
        <div class="chat-interface">
            <div class="brand-header">
                <img src="${safeGet(config, 'branding.logo', '')}" alt="${safeGet(config, 'branding.name', 'Chat Assistant')}">
                <span>${safeGet(config, 'branding.name', 'Chat Assistant')}</span>
                <button class="close-button">×</button>
            </div>
            <div class="chat-messages"></div>
            <div class="chat-input">
                <textarea placeholder="Type your message here..." rows="1"></textarea>
                <button type="submit">Send</button>
            </div>
            <div class="chat-footer">
                <a href="${safeGet(config, 'branding.poweredBy.link', 'https://legalgpts.com')}" target="_blank" rel="noopener noreferrer">
                    ${safeGet(config, 'branding.poweredBy.text', 'Powered by legalGPTs')}
                </a>
            </div>
        </div>
    `;

    chatContainer.innerHTML = newConversationHTML + chatInterfaceHTML;

    const toggleButton = document.createElement('button');
    toggleButton.className = `chat-toggle${safeGet(config, 'style.position', 'right') === 'left' ? ' position-left' : ''}`;
    // Transparent chat icon with white stroke, gradient background stays on button
    toggleButton.innerHTML = `
        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" aria-hidden="true">
            <path d="M12 2C6.5 2 2 6.5 2 12c0 1.8.5 3.5 1.3 5L2.5 21.5 7 20.7A10 10 0 1 0 12 2z"
                  fill="none" stroke="white" stroke-width="2"/>
        </svg>`;

    widgetContainer.appendChild(chatContainer);
    widgetContainer.appendChild(toggleButton);
    document.body.appendChild(widgetContainer);

    const newChatBtn = chatContainer.querySelector('.new-chat-btn');
    const chatInterface = chatContainer.querySelector('.chat-interface');
    const messagesContainer = chatContainer.querySelector('.chat-messages');
    const textarea = chatContainer.querySelector('textarea');
    const sendButton = chatContainer.querySelector('button[type="submit"]');

    function generateUUID() { return (crypto && crypto.randomUUID) ? crypto.randomUUID() : String(Date.now()) + Math.random().toString(16).slice(2); }
    let currentSessionId = '';

    // Markdown → HTML formatter (tight lists; no extra <br> inside <li>)
    function formatBotMessage(text) {
        if (!text) return '';
        let html = String(text).replace(/\r\n/g, '\n');

        // Links
        html = html.replace(/\[(.*?)\]\((https?:\/\/[^\s)]+)\)/g, '<a href="$2" target="_blank" rel="noopener noreferrer">$1</a>');

        // Bold and italics
        html = html.replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>');
        html = html.replace(/(^|[^*])\*(?!\*)(.*?)\*(?!\*)/g, '$1<em>$2</em>');

        // Ordered list blocks
        html = html.replace(/(^|\n)(\d+\.\s+.+(?:\n\d+\.\s+.+)*)/g, (match, prefix, block) => {
            const items = block.split('\n')
                .map(line => line.replace(/^\d+\.\s+/, '').trim())
                .filter(Boolean)
                .map(content => `<li>${content}</li>`)
                .join('');
            return `${prefix}<ol>${items}</ol>`;
        });

        // Unordered list blocks
        html = html.replace(/(^|\n)(-\s+.+(?:\n-\s+.+)*)/g, (match, prefix, block) => {
            const items = block.split('\n')
                .map(line => line.replace(/^-+\s+/, '').trim())
                .filter(Boolean)
                .map(content => `<li>${content}</li>`)
                .join('');
            return `${prefix}<ul>${items}</ul>`;
        });

        // Normalize excessive blank lines
        html = html.replace(/\n{3,}/g, '\n\n');

        // Wrap remaining text blocks as paragraphs
        html = html.split('\n\n').map(seg => {
            const trimmed = seg.trim();
            if (!trimmed) return '';
            if (trimmed.startsWith('<ul>') || trimmed.startsWith('<ol>')) return trimmed;
            return `<p>${trimmed.replace(/\n/g, '<br>')}</p>`;
        }).join('');

        return `<div>${html}</div>`;
    }

    async function sendMessage(message) {
        const userMessageDiv = document.createElement('div');
        userMessageDiv.className = 'chat-message user';
        userMessageDiv.textContent = message;
        messagesContainer.appendChild(userMessageDiv);
        messagesContainer.scrollTop = messagesContainer.scrollHeight;

        const body = {
            action: "sendMessage",
            sessionId: currentSessionId,
            route: safeGet(config, 'webhook.route', ''),
            chatInput: message,
            metadata: { userId: "" }
        };

        try {
            const response = await fetch(safeGet(config, 'webhook.url', ''), {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(body)
            });
            const data = await response.json();

            const botMessageDiv = document.createElement('div');
            botMessageDiv.className = 'chat-message bot';
            botMessageDiv.innerHTML = formatBotMessage(Array.isArray(data) ? data[0]?.output : data?.output);
            messagesContainer.appendChild(botMessageDiv);
            messagesContainer.scrollTop = messagesContainer.scrollHeight;
        } catch (error) {
            console.error('Webhook error:', error);
            const botMessageDiv = document.createElement('div');
            botMessageDiv.className = 'chat-message bot';
            botMessageDiv.textContent = 'Sorry—there was a problem reaching the server.';
            messagesContainer.appendChild(botMessageDiv);
        }
    }

    // Events
newChatBtn.addEventListener('click', () => {
  // Hide ONLY the welcome header (the first brand-header in the container)
  const headers = chatContainer.querySelectorAll('.brand-header');
  if (headers.length) {
    headers[0].style.display = 'none';      // welcome view header (top-level)
  }

  // Show the chat interface and make sure its header is visible
  const chatHeader = chatContainer.querySelector('.chat-interface .brand-header');
  if (chatHeader) {
    chatHeader.style.display = '';           // or 'flex' if you prefer: chatHeader.style.display = 'flex';
  }

  const welcome = chatContainer.querySelector('.new-conversation');
  if (welcome) {
    welcome.style.display = 'none';
  }

  chatInterface.classList.add('active');
  currentSessionId = generateUUID();
});

    sendButton.addEventListener('click', () => {
        const message = textarea.value.trim();
        if (message) {
            sendMessage(message);
            textarea.value = '';
        }
    });

    textarea.addEventListener('keypress', (e) => {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            const message = textarea.value.trim();
            if (message) {
                sendMessage(message);
                textarea.value = '';
            }
        }
    });

    toggleButton.addEventListener('click', () => {
        chatContainer.classList.toggle('open');
    });

    chatContainer.querySelectorAll('.close-button').forEach(button => {
        button.addEventListener('click', () => chatContainer.classList.remove('open'));
    });

})();
