<?php
/**
 * Plugin Name:  LegalGPTs Chat Widget
 * Description:  Injects the LegalGPTs chat widget in the footer. Admin can configure webhook, brand colors, logo, company name, greeting & secondary messages (HTML allowed), and choose where the JS is hosted (Bundled / Uploads / Remote). STRICT loader: no fallbacks.
 * Version:      1.2.5
 * Author:       LegalGPTs
 * License:      GPLv2 or later
 * Text Domain:  legalgpts-chat-widget
 */

if (!defined('ABSPATH')) { exit; }

class LegalGPTs_Chat_Widget_Plugin {
    const VERSION        = '1.2.5';
    const OPTION_KEY     = 'lgpts_chat_options';
    const UPLOADS_SUBDIR = 'legalgpts-chat-widget';
    const BUNDLED_JS_REL = 'assets/chat-widget.js';

    const DEFAULTS = [
        'webhook_url'       => '',
        'primary_color'     => '#002657',
        'secondary_color'   => '#00ffff',
        'hosting_mode'      => 'plugin',   // plugin | uploads | remote
        'script_url'        => 'https://example.com/chat-widget.js',
        'logo_url'          => '',
        'company_name'      => '',
        'uploads_url'       => '',
        'welcome_text_html' => 'Hi 👋, how can we help?',
        'response_text_html'=> 'We typically respond right away',
    ];

    public function __construct() {
        register_activation_hook(__FILE__, [$this, 'on_activate']);
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'admin_assets']);
        add_action('admin_notices', [$this, 'maybe_admin_notice']);
        add_action('wp_footer', [$this, 'print_config_block'], 5);
        add_action('wp_footer', [$this, 'print_loader_block'], 6);
    }

    public function on_activate() {
        $opts = get_option(self::OPTION_KEY);
        if (!is_array($opts)) { $opts = []; }
        $opts = array_merge(self::DEFAULTS, $opts);

        if (empty($opts['logo_url'])) {
            if (function_exists('has_site_icon') && has_site_icon()) {
                $opts['logo_url'] = esc_url_raw(get_site_icon_url(204));
            }
        }
        if (empty($opts['company_name'])) {
            $opts['company_name'] = get_bloginfo('name');
        }
        if ($opts['hosting_mode'] === 'uploads') {
            $opts['uploads_url'] = $this->ensure_uploads_script();
        }
        update_option(self::OPTION_KEY, $opts);
    }

    /** ---------------- Settings ---------------- */
    public function add_settings_page() {
        add_options_page(
            __('LegalGPTs Chat Widget', 'legalgpts-chat-widget'),
            __('LegalGPTs Chat Widget', 'legalgpts-chat-widget'),
            'manage_options',
            'legalgpts-chat-widget',
            [$this, 'render_settings_page']
        );
    }

    public function register_settings() {
        register_setting('lgpts_chat_group', self::OPTION_KEY, ['sanitize_callback' => [$this, 'sanitize_options']]);

        add_settings_section(
            'lgpts_chat_main',
            __('Widget Settings', 'legalgpts-chat-widget'),
            function () {
                echo '<p>' . esc_html__('Configure the chat widget. STRICT loader uses only the selected hosting location (no fallbacks).', 'legalgpts-chat-widget') . '</p>';
            },
            'legalgpts-chat-widget'
        );

        $fields = [
            'webhook_url'         => __('Webhook URL', 'legalgpts-chat-widget'),
            'primary_color'       => __('Primary Color', 'legalgpts-chat-widget'),
            'secondary_color'     => __('Secondary Color', 'legalgpts-chat-widget'),
            'hosting_mode'        => __('Script Hosting Location', 'legalgpts-chat-widget'),
            'script_url'          => __('Remote Script URL (used if hosting is "remote")', 'legalgpts-chat-widget'),
            'logo_url'            => __('Logo URL (defaults to Site Icon if empty)', 'legalgpts-chat-widget'),
            'company_name'        => __('Company Name (defaults to Site Name)', 'legalgpts-chat-widget'),
            'welcome_text_html'   => __('Greeting Message (HTML allowed)', 'legalgpts-chat-widget'),
            'response_text_html'  => __('Secondary Message (HTML allowed)', 'legalgpts-chat-widget'),
        ];

        foreach ($fields as $key => $label) {
            add_settings_field(
                "lgpts_{$key}",
                esc_html($label),
                [$this, 'render_field'],
                'legalgpts-chat-widget',
                'lgpts_chat_main',
                ['key' => $key]
            );
        }
    }

    public function sanitize_options($input) {
        $output = get_option(self::OPTION_KEY, self::DEFAULTS);

        $output['webhook_url']     = isset($input['webhook_url']) ? esc_url_raw(trim($input['webhook_url'])) : $output['webhook_url'];
        $primary   = isset($input['primary_color']) ? sanitize_hex_color($input['primary_color']) : $output['primary_color'];
        $secondary = isset($input['secondary_color']) ? sanitize_hex_color($input['secondary_color']) : $output['secondary_color'];
        $output['primary_color']   = $primary   ? $primary   : $output['primary_color'];
        $output['secondary_color'] = $secondary ? $secondary : $output['secondary_color'];

        $mode = isset($input['hosting_mode']) ? sanitize_text_field($input['hosting_mode']) : $output['hosting_mode'];
        if (!in_array($mode, ['plugin','uploads','remote'], true)) { $mode = 'plugin'; }
        $output['hosting_mode'] = $mode;

        $output['script_url']   = isset($input['script_url']) ? esc_url_raw(trim($input['script_url'])) : $output['script_url'];
        $output['logo_url']     = isset($input['logo_url']) ? esc_url_raw(trim($input['logo_url'])) : $output['logo_url'];
        $output['company_name'] = isset($input['company_name']) ? sanitize_text_field($input['company_name']) : $output['company_name'];

        $allowed = wp_kses_allowed_html('post');
        $output['welcome_text_html']  = isset($input['welcome_text_html'])  ? wp_kses(trim($input['welcome_text_html']), $allowed)  : $output['welcome_text_html'];
        $output['response_text_html'] = isset($input['response_text_html']) ? wp_kses(trim($input['response_text_html']), $allowed) : $output['response_text_html'];

        if (empty($output['logo_url']) && function_exists('has_site_icon') && has_site_icon()) {
            $output['logo_url'] = esc_url_raw(get_site_icon_url(204));
        }
        if (empty($output['company_name'])) {
            $output['company_name'] = get_bloginfo('name');
        }

        if ($output['hosting_mode'] === 'uploads') {
            $output['uploads_url'] = $this->ensure_uploads_script();
        } else {
            $output['uploads_url'] = '';
        }

        return $output;
    }

    public function render_field($args) {
        $opts = get_option(self::OPTION_KEY, self::DEFAULTS);
        $key  = $args['key'];
        $val  = isset($opts[$key]) ? $opts[$key] : '';

        if ($key === 'logo_url' && empty($val) && function_exists('has_site_icon') && has_site_icon()) {
            $val_hint = get_site_icon_url(204);
            echo '<p style="margin:0 0 6px 0;"><em>' . esc_html__('Using Site Icon by default:', 'legalgpts-chat-widget') . ' ' . esc_url($val_hint) . '</em></p>';
        }
        if ($key === 'company_name' && empty($val)) {
            echo '<p style="margin:0 0 6px 0;"><em>' . esc_html__('Using Site Name by default:', 'legalgpts-chat-widget') . ' ' . esc_html(get_bloginfo('name')) . '</em></p>';
        }

        switch ($key) {
            case 'primary_color':
            case 'secondary_color':
                printf('<input type="text" id="lgpts_%2$s" class="regular-text lgpts-color" name="%1$s[%2$s]" value="%3$s" />',
                    esc_attr(self::OPTION_KEY), esc_attr($key), esc_attr($val));
                echo '<p class="description" style="margin-top:4px;">' . esc_html__('Use a hex color (e.g., #002657).', 'legalgpts-chat-widget') . '</p>';
                break;

            case 'hosting_mode':
                $plugin_checked  = checked($val ?: 'plugin', 'plugin', false);
                $uploads_checked = checked($val, 'uploads', false);
                $remote_checked  = checked($val, 'remote', false);
                echo '<label><input type="radio" name="' . esc_attr(self::OPTION_KEY) . '[hosting_mode]" value="plugin" ' . $plugin_checked . '> ' . esc_html__('Bundled in Plugin (strict)', 'legalgpts-chat-widget') . '</label><br>';
                echo '<label><input type="radio" name="' . esc_attr(self::OPTION_KEY) . '[hosting_mode]" value="uploads" ' . $uploads_checked . '> ' . esc_html__('Uploads directory (strict)', 'legalgpts-chat-widget') . '</label><br>';
                echo '<label><input type="radio" name="' . esc_attr(self::OPTION_KEY) . '[hosting_mode]" value="remote" ' . $remote_checked . '> ' . esc_html__('Remote URL (strict)', 'legalgpts-chat-widget') . '</label>';
                if (!empty($opts['uploads_url'])) {
                    echo '<p class="description" style="margin-top:6px;">' . sprintf(esc_html__('Uploads script URL: %s', 'legalgpts-chat-widget'), '<code>' . esc_url($opts['uploads_url']) . '</code>') . '</p>';
                }
                break;

            case 'logo_url':
                printf('<input type="text" id="lgpts_logo_url" class="regular-text" name="%1$s[%2$s]" value="%3$s" /> ',
                    esc_attr(self::OPTION_KEY), esc_attr($key), esc_attr($val));
                echo '<button type="button" class="button" id="lgpts_logo_upload_btn">' . esc_html__('Select Image', 'legalgpts-chat-widget') . '</button>';
                $preview = $val ? esc_url($val) : ((function_exists('has_site_icon') && has_site_icon()) ? esc_url(get_site_icon_url(128)) : '');
                if ($preview) {
                    echo '<div style="margin-top:8px;"><img id="lgpts_logo_preview" src="' . esc_url($preview) . '" style="max-width:120px;height:auto;border:1px solid #ddd;padding:4px;background:#fff;" alt="Logo preview"></div>';
                } else {
                    echo '<div style="margin-top:8px;"><img id="lgpts_logo_preview" src="" style="display:none;max-width:120px;height:auto;border:1px solid #ddd;padding:4px;background:#fff;" alt="Logo preview"></div>';
                }
                break;

            case 'welcome_text_html':
            case 'response_text_html':
                printf('<textarea id="lgpts_%2$s" class="large-text code" rows="3" name="%1$s[%2$s]">%3$s</textarea>',
                    esc_attr(self::OPTION_KEY), esc_attr($key), esc_textarea($val));
                echo '<p class="description" style="margin-top:4px;">' . esc_html__('You can use HTML (e.g., <strong>, <em>, <a>, <br>).', 'legalgpts-chat-widget') . '</p>';
                break;

            default:
                printf('<input type="text" id="lgpts_%2$s" class="regular-text" name="%1$s[%2$s]" value="%3$s" />',
                    esc_attr(self::OPTION_KEY), esc_attr($key), esc_attr($val));
                break;
        }
    }

    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('LegalGPTs Chat Widget', 'legalgpts-chat-widget'); ?></h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('lgpts_chat_group');
                do_settings_sections('legalgpts-chat-widget');
                submit_button(__('Save Changes', 'legalgpts-chat-widget'));
                ?>
            </form>
            <hr>
            <p><strong><?php esc_html_e('Tips:', 'legalgpts-chat-widget'); ?></strong>
                <?php esc_html_e('Leave Logo URL empty to use your Site Icon (favicon). Leave Company Name empty to use your Site Name.', 'legalgpts-chat-widget'); ?>
            </p>
        </div>
        <?php
    }

    /** ---------------- Admin Assets ---------------- */
    public function admin_assets($hook) {
        $is_settings_page = ($hook === 'settings_page_legalgpts-chat-widget')
            || (isset($_GET['page']) && $_GET['page'] === 'legalgpts-chat-widget');

        if (!$is_settings_page) { return; }

        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        wp_enqueue_media();

        $handle = 'lgpts-chat-admin';
        wp_register_script($handle, false, array('jquery', 'wp-color-picker', 'media-editor'), '1.1', true);

        $inline = <<<JS
(function($){
  $(function(){
    $('.lgpts-color').wpColorPicker();
    var \$btn = $('#lgpts_logo_upload_btn');
    if (!\$btn.length) return;
    var frame = null;
    \$btn.on('click', function(e){
      e.preventDefault();
      if (!frame) {
        frame = wp.media({
          title: 'Select or Upload Logo',
          library: { type: ['image'] },
          button: { text: 'Use this logo' },
          multiple: false
        });
        frame.on('select', function(){
          var sel = frame.state().get('selection');
          if (!sel || sel.length === 0) return;
          var attachment = sel.first().toJSON();
          $('#lgpts_logo_url').val(attachment.url);
          var \$prev = $('#lgpts_logo_preview');
          if (\$prev.length) { \$prev.attr('src', attachment.url).css('display','block'); }
        });
      }
      frame.open();
    });
  });
})(jQuery);
JS;

        wp_enqueue_script($handle);
        wp_add_inline_script($handle, $inline, 'after');
    }

    /** ---------------- Admin Notices ---------------- */
    public function maybe_admin_notice() {
        if (!current_user_can('manage_options')) { return; }
        $screen = get_current_screen();
        if (!$screen || $screen->id !== 'settings_page_legalgpts-chat-widget') { return; }

        $opts = get_option(self::OPTION_KEY, self::DEFAULTS);
        $mode = isset($opts['hosting_mode']) ? $opts['hosting_mode'] : 'plugin';

        if ($mode === 'uploads' && empty($opts['uploads_url'])) {
            echo '<div class="notice notice-error"><p><strong>LegalGPTs Chat Widget:</strong> Uploads mode selected but the script was not copied to uploads. Check file permissions for <code>wp-content/uploads/</code> and click Save again.</p></div>';
        }
        if ($mode === 'remote' && empty($opts['script_url'])) {
            echo '<div class="notice notice-error"><p><strong>LegalGPTs Chat Widget:</strong> Remote URL mode selected but no Script URL is set.</p></div>';
        }
    }

    /** ---------------- Utilities ---------------- */
    private function ensure_uploads_script() {
        $bundled = plugin_dir_path(__FILE__) . self::BUNDLED_JS_REL;
        if (!file_exists($bundled)) { return ''; }

        $uploads = wp_upload_dir();
        if (!empty($uploads['error'])) { return ''; }

        $dir = trailingslashit($uploads['basedir']) . self::UPLOADS_SUBDIR;
        if (!is_dir($dir)) { wp_mkdir_p($dir); }
        $target = trailingslashit($dir) . 'chat-widget.js';

        if (!function_exists('WP_Filesystem')) { require_once ABSPATH . 'wp-admin/includes/file.php'; }
        WP_Filesystem();
        global $wp_filesystem;
        $data = file_get_contents($bundled);
        if ($wp_filesystem && is_object($wp_filesystem)) {
            $wp_filesystem->put_contents($target, $data, FS_CHMOD_FILE);
        } else {
            @file_put_contents($target, $data);
        }

        $url = trailingslashit($uploads['baseurl']) . self::UPLOADS_SUBDIR . '/chat-widget.js';
        return esc_url_raw($url);
    }

    /** ---------------- Front-end ---------------- */
    public function print_config_block() {
        if (is_admin()) { return; }
        $opts = get_option(self::OPTION_KEY, self::DEFAULTS);

        $logo_url = !empty($opts['logo_url'])
            ? esc_url($opts['logo_url'])
            : ((function_exists('has_site_icon') && has_site_icon()) ? esc_url(get_site_icon_url(204)) : '');

        $company_name = !empty($opts['company_name']) ? $opts['company_name'] : get_bloginfo('name');

        $welcome = $opts['welcome_text_html'];
        $response = $opts['response_text_html'];

        $config = [
            'webhook' => [
                'url'   => $opts['webhook_url'],
                'route' => '',
            ],
            'branding' => [
                'logo' => $logo_url,
                'name' => $company_name,
                'welcomeText' => $welcome,
                'responseTimeText' => $response,
                'poweredBy' => [
                    'text' => 'Powered by legalGPTs',
                    'link' => 'https://legalgpts.com',
                ],
            ],
            'style' => [
                'primaryColor'   => $opts['primary_color'],
                'secondaryColor' => $opts['secondary_color'],
                'position'       => 'right',
                'backgroundColor'=> '#ffffff',
                'fontColor'      => '#343741',
            ],
        ];

        echo "<script>window.ChatWidgetConfig = " . wp_json_encode($config, JSON_UNESCAPED_SLASHES) . ";</script>
";
    }

    public function print_loader_block() {
        if (is_admin()) { return; }
        $opts = get_option(self::OPTION_KEY, self::DEFAULTS);
        $mode = isset($opts['hosting_mode']) ? $opts['hosting_mode'] : 'plugin';

        $src = '';
        if ($mode === 'plugin') {
            $src = plugins_url(self::BUNDLED_JS_REL, __FILE__);
        } elseif ($mode === 'uploads') {
            $src = !empty($opts['uploads_url']) ? $opts['uploads_url'] : '';
        } else { // remote
            $src = !empty($opts['script_url']) ? $opts['script_url'] : '';
        }

        $ver = self::VERSION . '-' . md5(json_encode($opts));

        if (empty($src)) {
            echo "<!-- LegalGPTs Chat Widget STRICT loader: no script URL for mode {$mode} -->
";
            echo "<script>console.error('LegalGPTs Chat Widget (strict): No script URL available for mode: " . esc_js($mode) . "');</script>
";
            return;
        }

        ?>
<script>
(function(){
    var url = <?php echo json_encode($src, JSON_UNESCAPED_SLASHES); ?>;
    var ver = <?php echo json_encode($ver); ?>;
    var s = document.createElement('script');
    s.src = url + (url.indexOf('?') === -1 ? '?ver=' + ver : '&ver=' + ver);
    s.defer = true; s.async = true;
    s.onerror = function(){ console.error('LegalGPTs Chat Widget (strict): Failed to load script:', url); };
    document.body.appendChild(s);
})();
</script>
<!-- LegalGPTs Chat Widget STRICT loader mode: <?php echo esc_html($mode); ?>; url: <?php echo esc_url($src); ?> -->
        <?php
    }
}

new LegalGPTs_Chat_Widget_Plugin();
